unit GMP;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, GMP3DAPI, DsgnIntf;

type
  TGMPCutdown = class(TComponent)
  private
    { Private declarations }
    MyHDC : HDC;
    MyHGC : HGLRC;
    pfd : TPIXELFORMATDESCRIPTOR;
    PixelFormat : Integer;
    Lights : array [0..7] of TLight;
    FBackR, FBackG, FBackB : Single;
    FFlush : Boolean;
    FShadeEnabled : Boolean;
    F3DRunning : Boolean;
    procedure SetBackR(BackR : Single);
    procedure SetBackG(BackG : Single);
    procedure SetBackB(BackB : Single);
    procedure SetSwapBuffer(SwapIt : BOOLEAN);
    procedure Initialise3D(Running : Boolean);
    procedure CalcNormal(var P1, P2, P3, Normal1 : TPoint3D);

  protected
    { Protected declarations }
  public
    { Public declarations }
    MyLeft, MyRight, MyTop, MyBottom, PolygonCount : Integer;
    GMPGold : TMaterial;
    GMPSilver : TMaterial;
    GMPPewter : TMaterial;
    GMPChalk : TMaterial;
    GMPBlue_Plastic : TMaterial;
    GMPShiny_Brass : TMaterial;
    Ambient_Default : TColor4;
    Diffuse_Default : TColor4;
    Specular_Default : TColor4;
    Mat_Ambient : TColor4;
    Mat_Diffuse : TColor4;
    Mat_Emission : TColor4;
    Mat_Shininess : Single;
    Mat_Specular : TColor4;
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
    procedure SwapBackBuffer;
    procedure SetViewPort(left, right, top, bottom : Integer);
    procedure ClearBuffer;
    procedure ClearModelMatrix;
    procedure SetRotation(X,Y,Z,Angle : Single);
    procedure SetPosition(X,Y,Z : Single);
    procedure SetLight(Enabled : Boolean; var Position : TPos4; var Ambient, Diffuse, Specular : TColor4);
    procedure SetBuiltInMaterial(Name : String);
    procedure LoadRawObject(filename : String; Number : Integer);
    procedure DrawRawObject(Number : Integer);
    procedure SetCamera(var Position, Rotation : TPoint3D);
    procedure EnableFogging(YesNo : Boolean);
    procedure GMPSphere(size : Double);
    procedure GMPCube(size : Double);
    procedure GMPOct(size : Double);
    procedure GMPTet(size : Double);
    procedure GMPDod(size : Double);
    procedure GMPIcos(size : Double);
    procedure GMPTorus(width, circ : Double);
    procedure GMPCone(width, height : Double);
    procedure GMPCylinder(width, height : Double);
  published
    { Published declarations }
    property Back_Red : Single read FBackR write SetBackR;
    property Back_Green : Single read FBackG write SetBackG;
    property Back_Blue : Single read FBackB write SetBackB;
    property SwapTheBuffer : Boolean read FFlush write SetSwapBuffer;
    property Start3D : Boolean read F3DRunning write Initialise3D;

  end;

procedure Register;

implementation

procedure TGMPCutdown.GMPCylinder(width, height : Double);
begin
     DrawCylinder(width, height);
end;

procedure TGMPCutdown.GMPCone(width, height : Double);
begin
     DrawCone(width, height);
end;

procedure TGMPCutdown.GMPIcos(size : Double);
begin
     DrawIcosahedron(size);
end;


procedure TGMPCutdown.GMPDod(size : Double);
begin
     DrawDodecahedran(size);
end;


procedure TGMPCutdown.GMPTet(size : Double);
begin
     DrawTetrahedron(size);
end;

procedure TGMPCutdown.GMPOct(size : Double);
begin
     DrawOctahedron(size);
end;

procedure TGMPCutdown.GMPTorus(width, circ : Double);
begin
     DrawTorus(width, circ);
end;

procedure TGMPCutdown.GMPSphere(size : Double);
begin
     DrawSphere(size);
end;

procedure TGMPCutdown.GMPCube(size : Double);
begin
     DrawCube(size);
end;

procedure TGMPCutdown.EnableFogging(YesNo : Boolean);
var
   FogColor : TColor4;
begin
If YesNo = True then
begin
     glEnable(GL_FOG);
     glFogi(GL_FOG_MODE, GL_LINEAR);
     FogColor.R := 0.6;
     FogColor.G := 0.6;
     FogColor.B := 0.6;
     FogColor.A := 1;
     glFogfv(GL_FOG_COLOR, @fogColor.r);
     glFogf(GL_FOG_END, 10);
     glFogf(GL_FOG_START, 0);
     glHint(GL_FOG_HINT, GL_FASTEST);
end
else glDisable(GL_FOG);


end;



procedure TGMPCutdown.SetBuiltInMaterial(Name : String);
var
Material : TMaterial;
begin
     If name = 'Gold' then Material := GMPMaterial1;
     If name = 'Red' then Material := GMPMaterial2;
     if (name <> 'Gold') and (name <> 'Red') then exit;
    glMaterialfv(GL_FRONT, GL_AMBIENT, @Material.Ambient);
    glMaterialfv(GL_FRONT, GL_DIFFUSE, @Material.Diffuse);
    glMaterialfv(GL_FRONT, GL_SPECULAR, @Material.Specular);
    glMaterialfv(GL_FRONT, GL_EMISSION, @Material.Emitted);
    glMaterialfv(GL_FRONT, GL_SHININESS, @Material.SHininess);

end;


procedure TGMPCutdown.DrawRawObject(Number : Integer);
begin
     glCallList(Number);
end;

PROCEDURE TGMPCutdown.LoadRawObject(filename : String; Number : Integer );
var
   Line : String;
   p1,p2,p3,n1 : TPoint3D;
   RawFile : System.Text;
   Counter : Integer;
begin
     Counter := 0;
     AssignFile(RawFile, Filename);
     Reset(RawFile);
     glNewList(Number, GL_COMPILE);
     glBegin(GL_TRIANGLES);

     While not EOF(RawFile) do
     begin
          Readln(RawFile, p1.X, p1.Y, P1.Z);
          Readln(RawFile, p2.X, p2.Y, p2.z);
          Readln(RawFile, p3.X, p3.Y, p3.Z);
          CalcNormal(p1,p2,p3,n1);
          glNormal3f(-n1.X,-n1.Y,-n1.Z);
          glVertex3f(p3.X, p3.Y, p3.Z);
          glVertex3f(p2.X, p2.Y, p2.Z);
          glVertex3f(p1.X, p1.Y, p1.Z);
          INC(cOUNTER);

     end;
          glEnd;
     glEndList;
     CloseFile(RawFile);
     PolygonCount := Counter;

end;


procedure TGMPCutdown.Initialise3D(Running : Boolean);
var
X : Integer;
Position : TPOs4;
Color1 : TMaterial;
begin
    If (F3DRunning = False) and (Running = True) then
    begin
         F3DRunning := True;
         pfd.nsize :=  40;
         pfd.nVersion := 1;
         pfd.dwFlags :=  PFD_DRAW_TO_WINDOW + PFD_SUPPORT_OPENGL + PFD_DOUBLEBUFFER;
         pfd.iPixelType := PFD_TYPE_RGBA;
         pfd.cColorBits := 24; pfd.cRedBits := 0; pfd.cRedShift := 0; pfd.cGreenBits := 0;
         pfd.cGreenShift := 0; pfd.cBlueBits := 0; pfd.cBlueShift := 0; pfd.cAlphaBits := 0;
         pfd.cAlphaShift := 0; pfd.cAccumBits := 0; pfd.cAccumRedBits := 0; pfd.cAccumGreenBits := 0;
         pfd.cAccumBlueBits := 0; pfd.cAccumAlphaBits := 0; pfd.cDepthBits := 32;
         pfd.cStencilBits := 0; pfd.cAuxBuffers := 0; pfd.iLayerType := PFD_MAIN_PLANE;
         pfd.iLayerType := 0;
         pfd.bReserved := 0;
         pfd.dwLayerMask := 0;
         pfd.dwVisiblemask := 0;
         pfd.dwDamageMask := 0;
         pixelFormat := ChoosePixelFormat(MyHDC, @pfd);
         SetPixelFormat(MyHdc, pixelFormat, @pfd);
         DescribePixelFormat(MyHDC, pixelFormat, sizeof(pfd), pfd);
         MyHgc := wglCreateContext(MyHDC);
         wglmakeCurrent(MyHDC, MYHGC);
         SwapBuffers(MyHDC);
         glEnable(GL_DEPTH_TEST);
         For X := 0 to 7 do
         begin
              Lights[X].Number := X;
              Lights[X].enabled := False;
              Lights[X].Position.X := 0;
              Lights[X].Position.Y := 0;
              Lights[X].Position.Z := 0;
              Lights[X].Position.W := 1;
              Lights[X].Ambient.R := 0.5;
              Lights[X].Ambient.G := 0.5;
              Lights[X].Ambient.B := 0.5;
              Lights[X].Ambient.A := 1;
              Lights[X].Diffuse.R := 0.8;
              Lights[X].Diffuse.G := 0.8;
              Lights[X].Diffuse.B := 0.8;
              Lights[X].Diffuse.A := 1;
              Lights[X].Specular.R := 1;
              Lights[X].Specular.G := 1;
              Lights[X].Specular.B := 1;
              Lights[X].Specular.A := 1;
         end;
         Ambient_Default := Lights[0].Ambient;
         Diffuse_Default := Lights[0].Diffuse;
         Specular_Default := Lights[0].Specular;
         Mat_Ambient.r := 0.2;
         Mat_Ambient.g := 0.2;
         Mat_Ambient.b := 0.2;
         Mat_Ambient.a := 1.0;
         Mat_Diffuse.r := 0.8;
         Mat_Diffuse.g := 0.8;
         Mat_Diffuse.b := 0.8;
         Mat_Diffuse.a := 1;
         Mat_Specular := Ambient_Default;
         Mat_Shininess := 0;
         Mat_Emission := Ambient_Default;
         glEnable(GL_LIGHTING);
         glShadeModel(GL_SMOOTH);
         glPolygonMode(GL_FRONT, GL_FILL);
         glCullFace(GL_BACK);
         glEnable(GL_CULL_FACE);
         glFrontFace(GL_CCW);
         glClearColor(FbackR, FBackG, FbackB, 1.0);

    end;
end;


procedure TGMPCutdown.CalcNormal(var P1, P2, P3, Normal1 : TPoint3D);
var
   a,b : array [0..2] of Single;
   length : double;
   normal : TPoint3D;
begin
     a[0] := (P2.X - P1.X);
     a[1] := (P2.Y - P1.Y);
     a[2] := (P2.Z - P1.Z);
     b[0] := (P3.X - P1.X);
     b[1] := (P3.Y - P1.Y);
     b[2] := (P3.Z - P1.Z);


     normal1.X := (a[1] * b[2]) - (a[2] * b[1]);
     normal1.Y := (a[2] * b[0]) - (a[0] * b[2]);
     normal1.Z := (a[0] * b[1]) - (a[1] * b[0]);

     length := sqrt((normal1.X * normal1.X) + (normal1.Y * normal1.Y) + (normal1.Z * normal1.Z));
     if length <> 0 then
     begin
          normal1.X := normal1.X / length;
          normal1.Y := normal1.Y / length;
          normal1.Z := normal1.Z / length;
     end;


end;

procedure TGMPCutdown.SetLight(Enabled : Boolean; var Position : TPos4; var Ambient, Diffuse, Specular : TColor4);
begin
     Lights[0].Ambient := Ambient;
     Lights[0].Diffuse := Diffuse;
     Lights[0].Specular := Specular;
     Lights[0].Enabled := Enabled;
     Lights[0].Position.X := Position.X;
     Lights[0].Position.Y := Position.Y;
     Lights[0].Position.Z := Position.Z;
     Lights[0].Position.W := Position.W;
     If Lights[0].Enabled = True then
     begin
             ClearModelMatrix;
             glLightfv(GL_LIGHT0, GL_AMBIENT, @Ambient);
             glLightfv(GL_LIGHT0, GL_Diffuse, @Diffuse);
             glLightfv(GL_LIGHT0, GL_Specular, @Specular);
             glLightfv(GL_LIGHT0, GL_POSITION, @Position);
             glEnable(GL_Light0);
     end;
     If Lights[0].Enabled = False then
     begin
          glDisable(GL_Light0);
     end;

end;


procedure TGMPCutdown.SetPosition(X,Y,Z : Single);
begin
     glTranslatef(X,Y,Z);
end;

procedure TGMPCutdown.SetRotation(X,Y,Z, Angle : Single);
begin
    glRotatef(Angle, X, Y, Z);
end;

procedure TGMPCutdown.ClearModelMatrix;
begin
     glMatrixMode(GL_MODELVIEW);
     glLoadIdentity;
end;

procedure TGMPCutdown.ClearBuffer;
begin
     glClear(GL_COLOR_BUFFER_BIT or GL_DEPTH_BUFFER_BIT);

end;


procedure TGMPCutdown.SetViewPort(Left, Right, Top, Bottom : Integer);
var
Aspect : Single;
begin
     glViewPort(Left,Top, Right, Bottom);
     MyLeft := Left;
     MyTop := Top;
     MyRight := Right;
     MyBottom := Bottom;
     glMatrixMode(GL_PROJECTION);
     glLoadIdentity;
     Aspect := (MyRight - MyLeft) / (MyBottom - MyTop);
     gluPerspective(65, Aspect, 1.0, 200.0);

end;
procedure TGMPCutdown.SetCamera(var Position, Rotation : TPoint3D);
var
aspect : single;
begin
     glMatrixMode(GL_Projection);
     glLoadIdentity;
     Aspect := (MyRight - MyLeft) / (MyBottom - MyTop);
     gluPerspective(65, Aspect, 1.0, 200.0);

     glRotatef(Rotation.X,1,0,0);
     glRotatef(Rotation.Y,0,1,0);
     glRotatef(Rotation.Z,0,0,1);
     glTranslatef(Position.X, Position.Y, Position.Z);
end;

procedure TGMPCutdown.SetSwapBuffer(SwapIt : BOOLEAN);
begin
     FFlush := False;
     SwapBackBuffer;
end;

procedure TGMPCutdown.SwapBackBuffer;
begin
     glFlush;
     SwapBuffers(MyHDC);
end;
procedure TGMPCutdown.SetBackR(BackR : Single);
begin
     FBackR := BackR;
     glClearColor(FbackR, FBackG, FbackB, 1.0);
     glClear(GL_COLOR_BUFFER_BIT or GL_DEPTH_BUFFER_BIT);
end;

procedure TGMPCutdown.SetBackG(BackG : Single);
begin
     FBackG := BackG;
     glClearColor(FbackR, FBackG, FbackB, 1.0);
     glClear(GL_COLOR_BUFFER_BIT or GL_DEPTH_BUFFER_BIT);
end;

procedure TGMPCutdown.SetBackB(BackB : Single);
begin
     FBackB := BackB;
     glClearColor(FbackR, FBackG, FbackB, 1.0);
     glClear(GL_COLOR_BUFFER_BIT or GL_DEPTH_BUFFER_BIT);
end;

destructor TGMPCutdown.Destroy;
begin
     inherited Destroy;
     wglDeleteContext(MyHGC);
     DeleteDC(MyHDC);

end;

constructor TGMPCutdown.Create(AOwner : TComponent);
begin
    inherited Create(AOwner);
    F3dRunning := False;
    MyHDC := GetDC(TForm(AOwner).Handle);
    GMPGold.Name := 'Gold';
    GMPGold.Specular.r := 1;
    GMPGold.Specular.g := 1;
    GMPGold.Specular.b := 1;
    GMPGold.Specular.a := 1;

    GMPGold.Ambient.r := 0.5;
    GMPGold.Ambient.g := 0.5;
    GMPGold.Ambient.b := 0;
    GMPGold.Ambient.a := 1;

    GMPGold.Diffuse.r := 0.6;
    GMPGold.Diffuse.g := 0.6;
    GMPGold.Diffuse.b := 0.0;
    GMPGold.Diffuse.a := 1;

    GMPGold.Shininess := 30;

    GMPGold.Emitted.r := 0.1;
    GMPGold.Emitted.g := 0.1;
    GMPGold.Emitted.b := 0;
    GMPGold.Emitted.a := 1;

    GMPSilver.Name := 'Silver';
    GMPSilver.Specular.r := 1;
    GMPSilver.Specular.g := 1;
    GMPSilver.Specular.b := 1;
    GMPSilver.Specular.a := 1;

    GMPSilver.Ambient.r := 0.4;
    GMPSilver.Ambient.g := 0.4;
    GMPSilver.Ambient.b := 0.4;
    GMPSilver.Ambient.a := 1;

    GMPSilver.Diffuse.r := 0.8;
    GMPSilver.Diffuse.g := 0.8;
    GMPSilver.Diffuse.b := 0.8;
    GMPSilver.Diffuse.a := 1;

    GMPSilver.Shininess := 20;

    GMPSilver.Emitted.r := 0.1;
    GMPSilver.Emitted.g := 0.15;
    GMPSilver.Emitted.b := 0.1;
    GMPSilver.Emitted.a := 1;






end;

procedure Register;
begin
  RegisterComponents('Trial', [TGMPCutdown]);

end;


end.
